% Copyright (c) 2024, Daniel Molina Prez
%
% Project Title: Non-dominated Sorting Genetic Algorithm II (NSGA-II) for
% image enhancement
%
% Publisher: Daniel Molina Prez
% 
% Developer: Daniel Molina Prez
% 
% Contact Info: danielmolinaperez90@gmail.com

clc;
clear;
close all;

% Population size
Np=50;             

% Number of variables
Nvar=2; 

% Number of objective functions
Nf=2; 

% Number of evaluations
Neval=10000;  

% Mutation probability
Pm=1/Nvar; % Used by Deb NSGAII

% Crossover probability
Pc=0.7;

% SBX parameter
Nc=5;   

% Shape parameter of polynomial mutation
Nm=5;

% Threshold for details (band-pass filter)
dth=0.1;

% Lower bounds
lb=[0 0];
 
% Upper bounds
ub=[10 10];

% Read the input image
input_image1 = imread('kodim01.png');

% Convert the image to double data type
input_image1 = im2double(input_image1);

% Convert image to grayscale
input_image = rgb2gray(input_image1);

entropy0=entropy(input_image);

% Initial population and evaluation
p=zeros(Np,Nvar);
VFOp=zeros(Np,Nf);

for i=1:Np
    p(i,:)=lb+(ub-lb).*rand(1,Nvar);
    VFOp(i,:)=-FO(p(i,:),input_image,dth);
end

iter=1;
while Np*iter<=Neval
    
    % Non-Dominated Sorting
    F = NonDominatedSorting(VFOp);
    
    % Calculate Crowding Distance
    C = CrowdingDistance(VFOp, F);
    
    % Sort Population
    [p, VFOp, F, C] = SortPopulation(p,VFOp,F,C);

    % Selection: binary tournaments (sorted population)
    tour=[randperm(Np)' randperm(Np)']; 
    winners(tour(:,1)<=tour(:,2),1)= tour(tour(:,1)<=tour(:,2),1);
    winners(tour(:,1)>tour(:,2),1)= tour(tour(:,1)>tour(:,2),2);       
    parents=p(winners,:);

    % Crossover: SBX
    Hijos=zeros(Np,Nvar);
    for i=1:2:Np-1
        if rand<=Pc
            U=rand;
            for j=1:Nvar
                P1=parents(i,j);
                P2=parents(i+1,j);
                beta=1+2/(P2-P1)*min([(P1-lb(j)),(ub(j)-P2)]);
                alpha=2-abs(beta)^-(Nc+1);
                if U<=1/alpha
                    beta_c=(U*alpha)^(1/(Nc+1));
                else
                    beta_c=(1/(2-U*alpha))^(1/(Nc+1));
                end
                hijo1(1,j)=0.5*((P1+P2)-beta_c*abs(P2-P1));
                hijo2(1,j)=0.5*((P1+P2)+beta_c*abs(P2-P1));
            end
        else
            hijo1=parents(i,:);
            hijo2=parents(i+1,:);
        end
        Hijos(i,:)=hijo1;
        Hijos(i+1,:)=hijo2;
    end
    
    % Mutation: polynomial  
    for i = 1:Np
        for j = 1:Nvar
            if rand <= Pm
                u = rand;
                if u <= 0.5
                    delta = (2*u)^(1/(Nm+1)) - 1;
                    
                    % Mutate the individual
                    Hijos(i,j) = Hijos(i,j) + delta * (Hijos(i,j) - lb(j));
                else
                    delta = 1 - (2*(1-u))^(1/(Nm+1));
                    
                    % Mutate the individual
                    Hijos(i,j) = Hijos(i,j) + delta * (ub(j)- Hijos(i,j));
                end              
            end
        end
    end
  
parfor i=1:Np  
    VFOh(i,:)=-FO(Hijos(i,:),input_image,dth); 
end

    % Merge
    Pex = [p;Hijos];
    VFOex = [VFOp;VFOh];
    
    % Eliminate repeated vectors from the front
   [VFOex,ia,~] =unique(VFOex,'rows');
   Pex=Pex(ia,:);
    
    % Non-Dominated Sorting
    F = NonDominatedSorting(VFOex);
    
    % Calculate Crowding Distance
    C = CrowdingDistance(VFOex, F);
    
    % Sort Population
    [Pex, VFOex, F, C] = SortPopulation(Pex,VFOex,F,C);
    
    % Truncate
    p = Pex(1:Np,:);
    VFOp = VFOex(1:Np,:);
    
    iter=iter+1
end
figure(3)
plot(VFOp(:,1),VFOp(:,2),'*r');grid on

% Obtain the minimum and maximum values of each objective on the Pareto front
minVals = min(VFOp, [], 1);
maxVals = max(VFOp, [], 1);

% Normalize the Pareto front so that values are in the range [0, 1]
normalizedVFOp = (VFOp - minVals) ./ (maxVals - minVals);

% Define the normalized utopia point (all values are 0)
utopiaPoint = zeros(1, size(VFOp, 2));

% Calculate the Euclidean distance of each point to the utopia point
distances = sqrt(sum((normalizedVFOp - utopiaPoint).^2, 2));

% Identify the point with the shortest distance to the utopia point
[~, closestIndex] = min(distances);
Vclosest = VFOp(closestIndex, :);
Pclosest = p(closestIndex, :);

c = [p(1,1) Pclosest(1) p(2,1)]; % contrast factor
th = [p(1,2) Pclosest(2) p(2,2)]; % threshold value 

%--------------------Contrast and detail (for tabulation)---------------------
for i = 1:3
    % Apply the modified sigmoid function to each color plane
    output_image = 1 ./ (1 + exp(c(i) * (th(i) - input_image)));

    % Rescale the enhanced planes to the range [0, 1]
    output_image = mat2gray(output_image);

    % Define a manual smoothing filter (average)
    filter_size = 5; % Filter size
    average_filter = ones(filter_size) / (filter_size^2);

    % Apply the smoothing filter to obtain the blurred image
    blurred_image = imfilter(output_image, average_filter, 'replicate');

    % Subtract the blurred image from the original: highlights details and edges
    detail_image = output_image - blurred_image; % now can go from 0 to 1

    % Everything below the threshold is not detail and is set to zero (band-pass filter)
    detail_image(abs(detail_image(:))<=dth)=0;

    % Define an enhancement factor to control the intensity of the enhancement
    enhancement_factor = 1;
    output_image = output_image + enhancement_factor * detail_image;

    % Calculate Entropy
    HI(i) = entropy(output_image);

    % Calculate Normalized Standard Deviation
    sigma(i) = std(output_image(:)) / 0.5;

    % Calculate Number of detail pixels
    NHF(i) = sum(abs(detail_image)>0,"all");

    % Calculate Intensity of detail pixels
    IHF(i) = sum(abs(detail_image(:)));

    % Structural Similarity Index Measure
    SSIM(i)=ssim(output_image,input_image);
end

% Create a matrix to store the results
tabla_de_parametros = [HI(:), sigma(:), NHF(:), IHF(:), -[VFOp(1,:);Vclosest;VFOp(2,:)], SSIM(:)];

% Print the results table
fprintf('Solution\t\tH(I)\t\tsigma\tNHF\t\t\tIHF\t\t\tf1\t\t\tf2\t\t\tSSIM\n');
for i = 1:3
    fprintf('Extreme %d\t%8.4f\t%8.4f\t%4d\t%8.4f\t%8.4f\t%8.4f\t%8.4f\n', ...
        i, tabla_de_parametros(i, 1), tabla_de_parametros(i, 2), ...
        tabla_de_parametros(i, 3), tabla_de_parametros(i, 4), ...
        tabla_de_parametros(i, 5), tabla_de_parametros(i, 6), ...
        tabla_de_parametros(i, 7));
end

%---------------------------------plot----------------------------------
% Create a new figure
fig = figure;

% Get the screen size
screenSize = get(0, 'ScreenSize');

% Adjust the figure size to occupy the full screen
set(fig, 'Position', screenSize);

subplot(2,2,1)
imshow(input_image1);
for i=1:3
    % Apply the modified sigmoid function to each color plane
    output_image = 1 ./ (1 + exp(c(i) * (th(i) - input_image1)));

    % Rescale the enhanced planes to the range [0, 1]
    output_image = mat2gray(output_image);

    % Define a smoothing filter (average) manually
    filter_size = 5; % Filter size
    average_filter = ones(filter_size) / (filter_size^2);

    % Apply the smoothing filter to obtain the blurred image
    blurred_image = imfilter(output_image, average_filter, 'replicate');

    % Subtract the blurred image from the original image: highlights details and edges
    detail_image = output_image - blurred_image;

    % Everything below the threshold is not detail and is set to zero (band-pass filter)
    detail_image(abs(detail_image(:))<=dth)=0;

    % Define an enhancement factor to control the intensity of the enhancement
    enhancement_factor = 1;
    final_image = output_image + enhancement_factor * detail_image;

    subplot(2,2,i+1)
    imshow(final_image);
    title('Image with contrast and detail');
end
%--------------------------------------------------------------------------

% Objective function
function [VFO]=FO(x,input_image,dth)

% Define the contrast factor and the threshold value
c = x(1); % contrast factor
th = x(2); % threshold value (you can adjust this value)

% Apply the modified sigmoid function to each color plane
output_image = 1 ./ (1 + exp(c * (th - input_image)));

% Rescale the enhanced planes to the range [0, 1]
output_image = mat2gray(output_image);

% Define a smoothing filter (average) manually
filter_size = 5; % Filter size
average_filter = ones(filter_size) / (filter_size^2);

% Apply the smoothing filter to obtain the blurred image
blurred_image = imfilter(output_image, average_filter, 'replicate');

% Subtract the blurred image from the original image: highlights details and edges
detail_image = output_image - blurred_image; 

% Everything below the threshold is not detail and is set to zero (band-pass filter)
detail_image(abs(detail_image(:))<=dth)=0; 

% Define an enhancement factor to control the intensity of the enhancement
enhancement_factor = 1;
output_image = output_image + enhancement_factor * detail_image;

% FO1: entropy and normalized standard deviation
VFO1 = entropy(output_image)*std(output_image(:))/0.5;

% FO2: Number of detail pixels and pixel intensity
VFO2 = sum(abs(detail_image)>0,"all")*log10(log10(sum(abs(detail_image(:)))+1)+1);

VFO=[VFO1 VFO2];

end